<?php
/**
 * Manages Enqueues of scripts and styles
 *
 * @category Base
 * @package  OxyProps
 * @author   Cédric Bontems <cedric@thewebforge.dev>
 * @license  https://www.gnu.org/licenses/gpl-2.0.html  GPL v2 or later
 * @link     https://oxyprops.com                       OxyProps Website
 * @since    0.1.0
 */

namespace OxyProps\Inc;

use OxyProps\Inc\Singleton;

/**
 * Manages Enqueues
 *
 * @since 0.1.0
 * @author Cédric Bontems <contact@oxyprops.com>
 */
class Enqueue extends Base_Controller {


	use Singleton;

	/**
	 * Enqueue actions to run during class registration
	 *
	 * @since 0.2.0
	 * @since 1.11.4 fix bug in conditional logic for enqueueing scripts and styles.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function register(): void {
		$helpers          = Helpers::get_instance();
		$current_features = isset( $this->oxyprops_current_settings['settings']['features'] ) ? $this->oxyprops_current_settings['settings']['features'] : array();
		if ( isset( $current_features['dequeue_gutenberg'] ) && true === $current_features['dequeue_gutenberg'] ) {
			$helpers->dequeue_gutenberg_global_styles();
		}
		$helpers->register_settings();
		$helpers->brx_mute();
		$helpers->oxy_mute();
		$helpers->register_selectors();
		$helpers->register_colors();
	}

	/**
	 * Load scripts
	 *
	 * @since 0.2.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function initialize(): void {
		add_action( 'oxygen_enqueue_ui_scripts', array( $this, 'enqueue_oxy_js' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_bricks_js' ), 12 );
		add_action( 'wp_head', array( $this, 'enqueue_head_meta' ), 1 );

		add_action( 'wp_enqueue_scripts', array( $this, 'enqueueFront' ) );
		if ( $this->oxyprops_current_settings['settings']['features']['editor_styles'] ) {
			add_action( 'enqueue_block_editor_assets', array( $this, 'enqueueFront' ) );
		}

		if ( isset( $this->oxyprops_current_settings['settings']['features']['elements'] )
				&& $this->oxyprops_current_settings['settings']['features']['elements'] ) {
			switch ( $this->oxyprops_builder ) {
				case 'Oxygen':
					add_action( 'oxygen_add_plus_sections', array( $this, 'register_section' ) );
					add_action( 'oxygen_add_plus_oxyprops_section_content', array( $this, 'register_subsection' ) );
					add_action( 'plugins_loaded', array( $this, 'my_oxygen_elements_init' ) );
					break;
				case 'Bricks':
					add_action( 'init', array( Helpers::get_instance(), 'register_bricks_elements' ), 11 );
					add_filter(
						'bricks/builder/standard_fonts',
						function ( $standard_fonts ) {
							$standard_fonts = array(
								'var(--o-font-sans)',
								'var(--o-font-serif)',
								'var(--o-font-mono)',
								'var(--o-font-atkinson)',
								'Arial',
								'Helvetica',
								'Helvetica Neue',
								'Times New Roman',
								'Times',
								'Georgia',
								'Courier New',
							);
							return $standard_fonts;
						}
					);
					add_filter( 'bricks/builder/i18n', array( Helpers::get_instance(), 'register_bricks_strings' ) );
					break;
				case 'Breakdance':
					// Enqueue Breakdance Elements.
					break;
			}
		}
	}

	/**
	 * Enqueue Head meta tags.
	 *
	 * @since 1.4.12
	 * @since 1.9.1 Add user control over this feature.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function enqueue_head_meta(): void {
		$defaults   = Init::get_instances()['defaults'];
		$helpers    = Init::get_instances()['helpers'];
		$theme      = isset( $this->oxyprops_current_settings['settings']['theme'] ) ? $this->oxyprops_current_settings['settings']['theme'] : $defaults->get_oxyprops_default_settings()['theme'];
		$user_wants = $this->oxyprops_current_settings['settings']['features']['meta_theme_color'];
		if ( ! $user_wants ) {
			return;
		}
		$user_color_choice = $this->oxyprops_current_settings['settings']['features']['meta_theme_color_choice'];
		if ( 0 === strpos( $user_color_choice, 'surface' ) ) {
			$swatch = 'canvas';
		} else {
			$swatch = 'link' === $user_color_choice ? 'links' : $user_color_choice;
		}
		if ( 'custom' === $user_color_choice ) {
			$light = $this->oxyprops_current_settings['settings']['features']['meta_theme_color_choice_custom_light'];
			$dark  = $this->oxyprops_current_settings['settings']['features']['meta_theme_color_choice_custom_dark'];
		} else {
			$user_colors = $this->oxyprops_current_settings['settings']['colors'];
			$dark_shade  = $theme[ $user_color_choice . '_dark' ];
			$light_shade = $theme[ $user_color_choice . '_light' ];
			$dark_hex    = $helpers::hslToHex(
				array(
					'h' => $user_colors[ $swatch ]['swatch'][ $dark_shade ]['h'],
					's' => $user_colors[ $swatch ]['swatch'][ $dark_shade ]['s'],
					'l' => $user_colors[ $swatch ]['swatch'][ $dark_shade ]['l'],
				)
			);
			$light_hex   = $helpers::hslToHex(
				array(
					'h' => $user_colors[ $swatch ]['swatch'][ $light_shade ]['h'],
					's' => $user_colors[ $swatch ]['swatch'][ $light_shade ]['s'],
					'l' => $user_colors[ $swatch ]['swatch'][ $light_shade ]['l'],
				)
			);
			if ( $theme['lock_mode'] ) {
				$dark  = '#' . $light_hex;
				$light = '#' . $light_hex;
			} else {
				$dark  = '#' . $dark_hex;
				$light = '#' . $light_hex;
			}
		}
		?>
		<!-- OxyProps Meta -->
		<meta name="theme-color" media="(prefers-color-scheme: light)" content="<?php echo esc_html( $light ); ?>" />
		<meta name="theme-color" media="(prefers-color-scheme: dark)" content="<?php echo esc_html( $dark ); ?>" />
		<!-- End OxyProps Meta -->
		<?php
	}

	/**
	 * Enqueue frontend specifics.
	 *
	 * @since 0.2.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function enqueue(): void {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueueFront' ) );
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueueFront' ) );
	}

	/**
	 * Manage enqueue to builder editor and front-end
	 *
	 * @since 0.2.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function enqueueFront(): void {
		$helpers           = Helpers::get_instance();
		$current_features  = isset( $this->oxyprops_current_settings['settings']['features'] ) ? $this->oxyprops_current_settings['settings']['features'] : array();
		$current_utilities = isset( $this->oxyprops_current_settings['settings']['utilities'] ) ? $this->oxyprops_current_settings['settings']['utilities'] : array();
		$user_styles       = '';

		// Load props.
		// TODO: Split props into individual packages.
		$script_args = include $this->oxyprops_path . 'build/oxyprops-props.asset.php';
		wp_enqueue_style( 'oxyprops-props', $this->oxyprops_url . 'build/oxyprops-props.css', array(), $script_args['version'], 'all' );

		switch ( $this->oxyprops_builder ) {
			case 'Oxygen':
				if ( $current_features['normalize'] ) {
					$script_args = include $this->oxyprops_path . 'build/oxyprops-normalize.asset.php';
					wp_enqueue_style( 'oxyprops-normalize', $this->oxyprops_url . 'build/oxyprops-normalize.css', array(), $script_args['version'], 'all' );
				}
				$script_args = include $this->oxyprops_path . 'build/oxyprops-oxygen.asset.php';
				wp_enqueue_style( 'oxyprops-oxygen', $this->oxyprops_url . 'build/oxyprops-oxygen.css', array(), $script_args['version'], 'all' );
				$user_styles = $helpers->get_oxy_settings();
				break;
			case 'Bricks':
				if ( $current_features['normalize'] ) {
					$script_args = include $this->oxyprops_path . 'build/oxyprops-bricks-normalize.asset.php';
					wp_enqueue_style( 'oxyprops-normalize', $this->oxyprops_url . 'build/oxyprops-bricks-normalize.css', array(), $script_args['version'], 'all' );
				}
				$script_args = include $this->oxyprops_path . 'build/oxyprops-bricks.asset.php';
				wp_enqueue_style( 'oxyprops-bricks', $this->oxyprops_url . 'build/oxyprops-bricks.css', array(), $script_args['version'], 'all' );
				$user_styles = $helpers->get_brx_settings();
				break;
			case 'Breakdance':
                // phpcs:disable
                // if ( $current_features['normalize'] ) {
                // wp_enqueue_style( 'oxyprops-normalize', $this->oxyprops_url . 'build/oxyprops-breakdance-normalize.css', array(), $this->oxyprops_version, 'all' );
                // }
                // wp_enqueue_style( 'oxyprops-breakdance', $this->oxyprops_url . 'build/oxyprops-breakdance.css', array(), $this->oxyprops_version, 'all' );
                // $user_styles = $helpers->get_bd_settings();
                // phpcs:ensable
                break;
        }

        $user_styles .= $helpers->get_user_color_choices() . $helpers->get_user_theme_choices() . $helpers->get_user_typography_choices();
        wp_add_inline_style('oxyprops-props', $user_styles);

        // Load utility framework.
        if (isset($current_utilities['all']) && true === $current_utilities['all']) {
			$script_args = include $this->oxyprops_path . 'build/oxyprops-bundle.asset.php';
            wp_enqueue_style('oxyprops-bundle', $this->oxyprops_url . 'build/oxyprops-bundle.css', array(), $script_args['version'], 'all');
        } else {
			// Enqueue user selected stylesheets.
            foreach ($current_utilities as $name => $value) {
				if ('all' === $name) {
					continue;
                }
                if ($current_utilities[$name]) {
					$script_args = include $this->oxyprops_path . 'build/oxyprops-' . $name . '.asset.php';
                    wp_enqueue_style('oxyprops-' . $name, $this->oxyprops_url . 'build/oxyprops-' . $name . '.css', array(), $script_args['version'], 'all');
                }
            }
        }
    }

    /**
     * Enqueue Oxygen scripts
     *
     * @since 0.2.0
     * @author Cédric Bontems <contact@oxyprops.com>
     *
     * @return void
     */
    public function enqueue_oxy_js(): void
    {
        if (!isset(get_option($this->oxyprops_slug)['license']['key'])) {
            return;
        }
        if (!is_user_logged_in() || !isset($_GET['ct_builder']) || false === $_GET['ct_builder']) { // phpcs:ignore
            return;
        }

        $current_features = $this->oxyprops_current_settings['settings']['features'];
        if ($current_features['context']) {
            $script_args = include($this->oxyprops_path . 'build/oxyprops-oxygen.asset.php');
            wp_enqueue_script('oxyprops-oxygen', $this->oxyprops_url . 'build/oxyprops-oxygen.js', $script_args['dependencies'], $script_args['version'], true);
            //phpcs:ignore
            // wp_enqueue_script( 'oxyprops-context', $this->oxyprops_url . 'build/oxyprops-context.js', array(), $this->oxyprops_version, true );
        }
        // phpcs:disable
        // wp_enqueue_script( 'oxyprops-app', $this->oxyprops_url . 'build/oxyprops-app.js', array(), $this->oxyprops_version, true );
        // wp_enqueue_script( 'oxyprops-act', $this->oxyprops_url . 'build/oxyprops-act.js', array(), $this->oxyprops_version, true );
        // phpcs:enable
		wp_add_inline_script(
			'oxyprops-oxygen',
			'const OP_SETTINGS = ' . wp_json_encode(
				array(
					'colors'     => $current_features['global_colors'],
					'stylesheet' => $current_features['stylesheet'],
				)
			),
			'before'
		);
	}

	/**
	 * Enqueue Bricks scripts
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function enqueue_bricks_js() {
		if ( ! isset( get_option( $this->oxyprops_slug )['license']['key'] ) ) {
			return;
		}

        if (!is_user_logged_in() || !isset($_GET['bricks']) || false === $_GET['bricks']) { // phpcs:ignore
			return;
		}

		$current_features = $this->oxyprops_current_settings['settings']['features'];
		if ( $current_features['context'] ) {
			wp_enqueue_script( 'oxyprops-bricks', $this->oxyprops_url . 'build/oxyprops-bricks.js', array(), $this->oxyprops_version, true );
		}

		wp_add_inline_script(
			'oxyprops-bricks',
			'const OP_SETTINGS = ' . wp_json_encode(
				array(
					'colors'     => $current_features['global_colors'],
					'stylesheet' => $current_features['stylesheet'],
					'sez'        => defined( 'STRUCTEEZY_AUTOLOAD_VERSION' ),
				)
			),
			'before'
		);
	}

	/**
	 * Register section to Oxygen Plus menu
	 *
	 * @since 1.0.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function register_section() {
		\CT_Toolbar::oxygen_add_plus_accordion_section( 'oxyprops', __( 'OxyProps', 'oxyprops-oxygen' ) );
	}

	/**
	 * Register SubSections to Oxygen Plus Menu
	 *
	 * @since 1.0.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function register_subsection() {
		?>

		<h2><?php esc_html_e( 'Visual', 'oxygen' ); ?></h2>
		<?php do_action( 'oxygen_add_plus_oxyprops_visual' ); ?>

		<h2><?php esc_html_e( 'Colors & Schemes', 'oxygen' ); ?></h2>
		<?php do_action( 'oxygen_add_plus_oxyprops_colors' ); ?>

		<h2><?php esc_html_e( 'Other', 'oxygen' ); ?></h2>
		<?php do_action( 'oxygen_add_plus_oxyprops_other' ); ?>

		<?php
	}

	/**
	 * Loads OxyProps custom elements
	 *
	 * @return void
	 */
	public function my_oxygen_elements_init() {

		if ( ! class_exists( 'OxygenElement' ) ) {
			return;
		}

		foreach ( glob( $this->oxyprops_path . 'includes/elements/oxygen/*.php' ) as $filename ) {
			include $filename;
		}
	}
}
